//---------------------------------------------------------------------------
// This demo program reads latitude and longitude values from a GPS receiver
// and displays them in the terminal window. The 4800 baud serial output from
// the GPS receiver is connected to the SERIN pin of the uM-FPU V3.1 chip.
// The sentence count and sentence type is displayed for each record, and the
// longitude and latitude values are displayed for the GPRMC sentence.
// The sentence types will vary depending on the GPS receiver used.
// The following example shows output from a Garmin receiver:
//
// @author Cam Thompson, Micromega Corporation
// <www.micromegacorp.com>
//
// @version
//  October 2, 2011
//  - updated for new SPI and FpuSerial libraries
//  December 15, 2008
//  - original version
//
//    GPSdemo1
//    uM-FPU V3.1.0
//
//    1:GPGLL
//    2:GPBOD
//    3:GPVTG
//    4:PGRME
//    5:PGRMZ
//    6:PGRMM
//    7:HCHDG
//    8:GPRTE
//    9:GPRMC: Lat = 7622.4609, Long = 7622.4609
//    (etc.)
//---------------------------------------------------------------------------

#include <SPI.h>
#include <Fpu.h>
#include <FpuSerial.h>

//------------------ uM-FPU register definitions ----------------------------

#define Latitude     10                 // latitude value
#define Longitude    11                 // longitude value

#define GPS_OK       0x80               // GPS sentence OK status

//-------------------- setup ------------------------------------------------

void setup()
{
  Serial.begin(9600);
  Serial.println("GPSdemo1");
  
  SPI.begin();
  Fpu.begin();
  
  if (Fpu.sync() == SYNC_CHAR)
    FpuSerial.printVersionln();
  else
  {
    Serial.println("FPU not detected");
    while(1) ; // stop if FPU not detected
  }  
}

//-------------------- loop -----------------------------------------------

void loop()
{
  byte status;
  
  // set FPU baud rate to 4800, enable NMEA input
  Fpu.write(SEROUT, 0, 5);
  Fpu.write(SERIN, 4);

  for (int cnt = 0; cnt < 100; cnt++)
  {
    Serial.println();
    Serial.print(cnt, DEC);
    Serial.print(':');

    Fpu.write(SERIN, 6);
    status = Fpu.readStatus();

    if (status == GPS_OK)
    {
      FpuSerial.printString(READSEL);

      Fpu.write(STRCMP);
      Fpu.writeString("GPRMC");
      status = Fpu.readStatus();
      
      if (status & STATUS_ZERO)
      {
        Fpu.write(SELECTA, Latitude);
        Fpu.write(STRFIELD, 4, STRTOF, FSET0);
        Fpu.write(SELECTA, Longitude);
        Fpu.write(STRFIELD, 6, STRTOF, FSET0);
        Fpu.write(STRSET, 0);

        Fpu.write(SELECTA, Latitude);
        Serial.print(" *** Lat = " );
        FpuSerial.printFloat(0);

        Fpu.write(SELECTA, Longitude);
        Serial.print(", Long = ");
        FpuSerial.printFloat(0);
      }
    }
    else
    {
      Serial.print(" *** data error ***");
    }
  }

  Serial.println("\r\nDone.");
  while(1) ;
}
